;; -*- mode: common-lisp; package: top-level.debug -*-
;;
;; copyright (c) 1993-2005 Franz Inc, Berkeley, CA  - All rights reserved.
;; copyright (c) 2002-2012 Franz Inc, Oakland, CA - All rights reserved.
;;
;; Licensed users of Allegro CL may include the following macro in their
;; product, provided that product is only compiled with a licensed Allegro
;; CL compiler.
;;
;; $Id: autozoom.cl,v 1.15 2007/04/17 21:27:35 layer Exp $

(eval-when (load eval) (require :tpl-debug))

(defpackage :top-level.debug
  (:use :common-lisp :excl)
  (:export #:with-auto-zoom-and-exit #:zoom))

(provide :autozoom)

(in-package :top-level.debug)

(defmacro with-auto-zoom-and-exit ((place &key (count 't) (all 't)
					       (exit 't) no-unwind)
				   &body body)
  "Generate a execution stack trace to PLACE, which can be a stream, `t' or
a pathname.  This macro is a wrapper for top-level :zoom command and is
intended to be used in application that want to report unexpected errors.

The COUNT and ALL keywords are passed to the :zoom command and are
documented with that command.

A non-nil value for EXIT cause the running application to terminate.  The
default value of EXIT is `t'.

A non-nil value for NO-UNWIND causes unwind-protects in stack frames above
the error to be ignored.  (This argument is simply passed to exit which
also accepts a NO-UNWIND keyword argument.)  NO-UNWIND is ignored if
EXIT is nil.  The default value of NO-UNWIND is nil (as it is for EXIT)."
  (let ((g-place (gensym))
	(g-count (gensym))
	(g-all (gensym))
	(g-exit (gensym))
	(g-no-unwind (gensym)))
    `(let ((,g-place ,place)
	   (,g-count ,count)
	   (,g-all ,all)
	   (,g-exit ,exit)
	   (,g-no-unwind ,no-unwind))
       (handler-bind
	   ((error (lambda (e)
		     (with-standard-io-syntax
		       (let ((*print-readably* nil)
			     (*print-miser-width* 40)
			     (*print-pretty* t)
			     (top-level:*zoom-print-circle* t)
			     (top-level:*zoom-print-level* nil)
			     (top-level:*zoom-print-length* nil))
			 (ignore-errors	;prevent recursion
			  (format *terminal-io* "~
~@<An unhandled error condition has been signalled:~3I ~a~I~:@>~%~%"
				  e))
			 (ignore-errors	;prevent recursion
			  (flet ((do-zoom (s)
				   (let ((*terminal-io* s)
					 (*standard-output* s))
				     (top-level:do-command "zoom"
				       :from-read-eval-print-loop nil
				       :count ,g-count :all ,g-all))))
			    (when (eq 't ,g-place)
			      (setq ,g-place *terminal-io*))
			    (if* (streamp ,g-place)
			       then (do-zoom ,g-place)
			       else (with-open-file (s ,g-place
						     :direction :output
						     :if-exists :supersede)
				      (do-zoom s)))))
			 (when ,g-exit
			   (exit 1 :no-unwind ,g-no-unwind)))))))
	 ,@body))))

(defun zoom (stream &rest zoom-command-args
	     &key (count t) (all t)
	     &allow-other-keys)
  "Generate a execution stack trace to STREAM.  This function is a wrapper
for top-level :zoom command and is intended to be used in application that
want to report unexpected errors.  See with-auto-zoom-and-exit.

This function is only useful when used in conjunction with HANDLER-BIND.
For example:

 (handler-bind
    ((error (lambda (condition)
	      ;; write info about CONDITION to a log file...
              (format *log-stream* \"Error in app: ~a~%\" condition)

	      ;; send a zoom to the log file, too
	      (top-level.debug:zoom *log-stream*))))
  (application-init-function))

The COUNT and ALL keywords are passed to the :zoom command and are
documented with that command."
  (with-standard-io-syntax
    (let ((*print-readably* nil)
	  (*print-miser-width* 40)
	  (*print-pretty* t)
	  (top-level:*zoom-print-circle* t)
	  (top-level:*zoom-print-level* nil)
	  (top-level:*zoom-print-length* nil)
	  (*terminal-io* stream)
	  (*standard-output* stream))
      (apply #'top-level:do-command "zoom"
	     :from-read-eval-print-loop nil
	     :count count :all all zoom-command-args))))
