#+(version= 8 2 :beta)
(sys:defpatch "aclstart" 1
  "v1: readtable fix for #!."
  :type :system
  :post-loadable nil)

;; -*- mode: common-lisp; package: excl -*-
;;
;; copyright (c) 1986-2005 Franz Inc, Berkeley, CA  - All rights reserved.
;; copyright (c) 2002-2012 Franz Inc, Oakland, CA - All rights reserved.
;;
;; Allegro Common Lisp
;; The application-customizable lisp startup code.
;;
;; This code is provided to licensed users of Allegro CL as an example
;; and for restricted reuse.
;;
;; Restricted Rights Legend
;; ------------------------
;; Use, duplication, and disclosure of the software, data and information
;; contained herein by any agency, department or entity of the U.S.
;; Government are subject to restrictions of Restricted Rights for
;; Commercial Software developed at private expense as specified in
;; DOD FAR Supplement 52.227-7013 (c) (1) (ii), as applicable.

;; This file contains the important lisp code that starts or restarts execution
;; of a lisp image.  This file is distributed in source form so application
;; writers can first understand the mechanisms, and then to customize them as
;; may sometimes be necessary for application delivery.
;;
;; Note carefully the restriction below concerning the code that prints
;; the Franz Inc copyright banner.

(in-package :excl)

;; This mechanism is used for emergency exit e.g. when *terminal-io* is broken.
;; Beware: If this hook is set to '_exit, a process-reset of the initial
;; lisp listener will cause lisp to exit silently and immediately!

(defvar *initial-lisp-listener-exit-hook* 'nil)

;; This function is wrapped around the start of every thread computation
;; automatically by the low-level thread startup code.
;; User code need not be concerned with it and should not modify it.
;; The source is included here to document the dynamic environment that is
;; automatically wrapped around every computation.

(declaim (ftype (function) private-exit-lisp))

(defvar *lisp-kill-values* ())

(defun throw-lisp-kill (code exit quiet)
  (setf (si::thread-state (current-thread)) :unwinding-for-exit)
  ;; Remember these values to be able to rethrow later. We don't care
  ;; about races between EXIT calls.
  (setq *lisp-kill-values* (list code exit quiet))
  (throw 'lisp-kill (values code exit quiet)))

(defun setup-required-thread-bindings (&aux vars vals)
  ;; When this function is called the
  ;; value of sys:*current-process* may still be nil.
  (progn
      (dolist (b *required-thread-bindings*) ;; [rfe11065]
	 (unless (member (car b) vars :test #'eq)
	   (push (car b) vars)
	   (push (eval (cdr b)) vals)))
      ;; Error handlers need to find these variables on every
      ;; stack group.  Don't make this list longer than it needs
      ;; to be.
      (progv vars vals
	 (setq vars nil vals nil)
	 ;; In addition to the normal prov-bound variables, *stack-pc*
	 ;; must be explicitly bound, because its locative needs to
	 ;; be on the stack (it could be panified, but that forces
	 ;; copying).  Perhaps in the future, especially if there
	 ;; are more of these variables whose locatives we are messing
	 ;; with, a compiler-macro can do this more generally with another
	 ;; declarative list.
	 
	 (let ((*byte-si* 0))

	   ;; Interrupts are disabled until this point in both a
	   ;; dumplisp restart and in a thread initial call so
	   ;; that no thread reset can happen until inside the
	   ;; catcher.
	   ;; bug1278 - smh 1jun90.
	   ;; The test for #'do-the-dump is a crock designed to treat
	   ;; dumplisp specially.  See .dumplisp in code/misc, and
	   ;; see bug2893 for more details.  - smh 14jan93.

	   (multiple-value-bind (code exit quiet)
               (catch 'lisp-kill
                 (thread-reset-catcher))
             (when (and (not exit) (si:scheduler-running-p))
	       ;; just return the value
	       (throw 'thread-return code))
	     (locally
		 (declare (ftype (function) private-exit-lisp))
               (private-exit-lisp (or code 0) :quiet quiet))))))
  ;; return a single value, even if the abort restart gets triggered
  nil)


;; thread-reset-catcher exits only by throwing and when it exits
;; thread state is :exhausted

(defun thread-reset-catcher ()
  (let ((ct (current-thread))
        (return-values))
    (loop
     (catch 'thread-reset
       (unwind-protect
            (restart-case
                ;; These two catches with top-level in their names are
                ;; simply thrown to, there is no accompanying state
                ;; change. In what circumstances they are not caught
                ;; by the normal catches, gmelis has no idea as of
                ;; 2010-07-28, but let's maintain compatiblity by
                ;; treating them as normal returns.
                ;;
                ;; Used for pop etc. from non-lisp-listener thread.
                (catch 'tpl::top-level-break-loop
                  ;; Used for reset from non-lisp-listener thread, or
                  ;; restart-function.
                  (catch :top-level-reset
                    (when (si::thread=>state ct :active :start)
                      (setq return-values '())
                      (setq return-values (multiple-value-list
                                              (thread-bind-and-call
                                               (si::thread-initial-bindings ct)
                                               (si::thread-initial-function ct)
                                               (si::thread-initial-args ct)))))))
              ;; bug7749
              ;;
              ;; This restart used to be in a caller of this function,
              ;; but to pass the unwind barrier the thread state must
              ;; be set up correctly.
              (abort ()
                :report "Abort entirely from this (lisp) process."
                (throw-lisp-kill nil nil nil)))
         ;; THREAD-STATE and to where we are throwing may be out of
         ;; sync. An example is when throwing to THREAD-RESET, a
         ;; PROCESS-KILL changes state to :UNWINDING-FOR-KILL and
         ;; schedules an interruption to throw to THREAD-KILL but this
         ;; interruption arrives too late. The one we trust is the
         ;; nicely CASed THREAD-STATE.
         ;;
         ;; This is a unwind barrier: it will cancel any ongoing
         ;; unwind and throw to a target depending on THREAD-STATE. It
         ;; relies on the fact that interrupts are disabled.
         (loop
          (assert *without-interrupts*)
          (cond
            ;; lisp-kill
            ((si::thread=>state ct :exhausted :unwinding-for-exit)
             (fast (setf (sys::thread-whostate (the sys::thread ct))
                         "Exiting"))
             (throw 'lisp-kill (values-list *lisp-kill-values*)))
            ;; kill
            ((si::thread=>state ct :exhausted :unwinding-for-kill)
             (fast (setf (sys::thread-whostate (the sys::thread ct))
                         "Killed"))
             (throw 'thread-kill nil))
            ;; reset
            ((or (si::thread=>state ct :start :unwinding-for-reset)
                 (and (si::thread-reset-action ct)
                      (si::thread=>state ct :start :active)))
             ;; Make sure that pending interruptions are thrown out on
             ;; a reset as they are on a kill followed by a reset.
             (setf (sys::thread-interrupts ct) nil
                   (sys::thread-urgent-interrupts ct) nil)
             ;; Auto reset is a shortcut, make sure process-join works
             ;; even if we don't exit and start a new thread.
             (si::note-thread-finish return-values)
             (throw 'thread-reset nil))
            ;; normal return
            ((si::thread=>state ct :exhausted :active)
             (throw 'thread-return (values-list return-values))))))))))

;; thread-bind-and-call accomodates the specified bindings, then
;; applies the function to the arguments

(defun thread-bind-and-call* (initial-bindings function args)
  #-smp
  (let ((ct (current-thread)))
    (unless (and (si::thread-run-reasons ct)
                 (null (si::thread-arrest-reasons ct)))
      ;; bug14646: a newly started thread that isn't active hangs up here
      (si::yield)))
  (if* (null initial-bindings)
     then (apply function args)
     else (let ((vars '(tpl::*initial-bindings-done*))
                (vals '(t)))
            (dolist (b initial-bindings)
              (unless (member (car b) vars :test #'eq)
                (push (car b) vars)
                (push (eval (cdr b)) vals)))
            (progv vars vals
              (setq vars nil vals nil)
              (apply function args)))))

(defun thread-bind-and-call (initial-bindings function args)
  ;; Now we can allow interrupts (bug14646), take care to disallow
  ;; them after we finish here (bug1278, bug19514).
  (assert *without-interrupts*)
  ;; bug20598: don't allow interrupts for all threads.
  (let ((process (sys::thread-process sys:*current-thread*)))
    (if (and process
             ;; We provide no way to turn off interrupts in the
             ;; initial process that is a for a while whose first
             ;; element is "Stub process".
             (not (vectorp process))
             (slot-value process 'si::without-interrupts))
        (thread-bind-and-call* initial-bindings function args)
        (with-interrupts
          #+smp
          ;; Let arrest and run reasons block the thread.
          (excl::run-interruptions)
          (thread-bind-and-call* initial-bindings function args)))))



;; This variable controls whether lisp startup tries to process .clinit files.
;; Upon startup its value is copied to the internal variable, which may be
;; altered by the standard command-line switch processing, see below.
(defvar *read-init-files* t)
(defvar *internal-read-init-files* t)

;; If this is non-NIL it is a function of zero arguments that will be called
;; before tpl:start-interactive-top-level.  It should be used to make any
;; initializations before the top-level.

(defvar *restart-init-function* nil)

;; If this is non-NIL it is a function of zero arguments that will be called
;; to start lisp execution instead of the regular read-eval-print loop.
;; The intention is that it would be the top level of a delivered application.
;; It should never return; if it does, results are unpredictable.

(defvar *restart-app-function* nil)

;; This is a list of functions of zero arguments that will be called when a
;; dumplisp image restarts.  These allow the restarting image to do necessary
;; cleanups.  This list is not altered by startup and remains in effect when
;; restarting a subsequent dumplisp, but the function immediately following
;; might be useful to add an action that runs just once.

(defvar *restart-actions* nil)

;; on the windows platform, this is the name of the imagefile we are running
(defvar *image-file* nil)

;; If this is NIL, then all errors from init files, -e/-ee forms, -L
;; files are caught. Errors from init files will cause an error and
;; exit the lisp, while errors from -e/-ee forms, and -L files will
;; turn into a warning warning in non-batch mode.
;;
;; It this is non-NIL, then errors will encountered in the above
;; situations will not be handled making it possible to get meaningful
;; backtraces with -batch and -backtrace-on-error; and an interactive
;; debug session without -batch.
;;
;; The command line option --debug-startup sets this to non-NIL.
(defvar *debug-startup* nil)

;; If this is non-NIL and *debug-startup* is NIL, then
;; maybe-handle-errors-in-startup will handle errors (turn them into
;; warnings in non-batch mode).
(defvar *startup-phase-name* nil)

(defmacro with-startup-phase ((format-control &rest args) &body body)
  (let ((cancel-this-startup-phase (gensym "cancel-this-startup-phase")))
    `(catch ',cancel-this-startup-phase
       (handler-bind ((error (lambda (err)
                               ;; Let errors from startup code
                               ;; through if *debug-startup*.
                               (unless (and *debug-startup*
                                            *startup-phase-name*)
                                 ;; Turning the error into a warn, or
                                 ;; resignaling it effectively muffles
                                 ;; -backtrace-on-error, special case
                                 ;; it because we are nice.
                                 (when *backtrace-on-error*
                                   (print-backtrace-on-error
                                    :from-read-eval-print-loop nil))
                                 (let ((*print-pretty* t))
                                   (funcall (if *batch-mode*
                                                #'error
                                                #'warn)
                                            "An error occurred (~a) during ~a"
                                            err *startup-phase-name*))
                                 ;; Want to get out even if it was
                                 ;; only a warn.
                                 (throw ',cancel-this-startup-phase nil)))))
         (let ((*startup-phase-name* (format nil ,format-control ,@args)))
           ,@body)))))

(eval-when (compile)
  (declaim (ftype (function) check-8-bit-input)))

(defvar *pll-file* nil)
(defun pll-file () *pll-file*)

;; Fixed for spr22470 -- cac 8aug00
(defun locale-from-string (locale-string
			   ;; rfe4968
			   &aux (*load-verbose* nil))
  (let ((locale (find-locale locale-string)))
    (if* locale
       thenret
       else (warn "~
Locale name `~a' references unknown external-format.  ~
Using `~a' instead."
		  locale-string (locale-name *locale*))
	    *locale*)))

#-mswindows (defvar .startup-script. nil)
(defvar *force-quiet-exit* nil)
(defvar *script* nil)

(eval-when (compile)
  (when (fboundp 'sys::expired)
    (pushnew :has-sys-expired *features* :test #'eq)))

(defvar *start-lisp-execution-initialized* nil)

(defun start-lisp-execution (reborn ;; [bug12247]
			     &aux (default-init-ef
				      ;; Set to default external-format
				      ;; before *locale* gets reset.
				      (find-external-format :default))
				  (no-listener-sleep-forever nil))

  (unless *start-lisp-execution-initialized*
    (setq *start-lisp-execution-initialized* t)
    (setq *locale* (find-locale "C"))

    ;; Here is where we set *locale* based on setlocale()'s return value:
    (ics-target-case
      (:+ics
       (setq *locale*
             (or (let ((locale-string
                        (or (sys:getenv "ACL_LOCALE")
                            (posix-setlocale
                             :lc-ctype
                             ;; Passing nil to posix-setlocale is equivalent to
                             ;; passing NULL to setlocale().
                             ;; On Windows, (discovered empirically),
                             ;; setlocale(..., NIL) returns "C" whereas
                             ;; setlocale(..., "") returns the actual Windows locale.
                             ;; Thus we special-case here
                             #| #+mswindows "" #-mswindows nil |#
                             ;; Update: 22dec06: Use setlocale(..., "") universally.
                             ;; bug16552
                             ""))))
                   (when locale-string
                     (locale-from-string locale-string)))
                 *locale*))))
    
    (sys::reset-make-temp-file-name-counters)

    (flush-all-logical-pathname-translations)

    (setq *clear-input-on-error*
          #-mswindows t
          ;; Make sure this gets set appropriately for build.exe (ie, set to nil):
          #+mswindows (ms-windows-based-app-p))

    (let ((s *initial-terminal-io*))
      ;; Bind all the standard streams to *initial-terminal-io*
      ;; so the startup code can be guaranteed a known I/O environment.

      (handler-bind
          ((error (lambda (c)
                    (unless (and *debug-startup*
                                 *startup-phase-name*)
                      (ignore-errors
                        (let ((*print-pretty* t))
                          (format s
                                  (newlinify-format-string
                                   "~
An unhandled error occurred during initialization: ~a~%")
                                  c))
                        (force-output s)
                        (exit-lisp -1 :quiet t))))))
          (with-standard-io-syntax
              (let ((*terminal-io* s)
                    (*standard-input* s)
                    (*standard-output* s)
                    (*query-io* s)
                    (*debug-io* s)
                    (*trace-output* s)
                    (*error-output* s)
                    (*print-pretty* *pprint-gravity*)
                    (*print-readably* nil))

                ;; [bug16571]: Fix dumplisp-generation-tick.
                (when (or (stream-input-fn s) (stream-output-fn s))
                  (setf (getf (slot-value s 'plist) :dumplisp-generation-tick)
                        *dumplisp-generation-tick*))
                
                ;; If the scheduler was running in the pre-dumplisp image,
                ;; it is necessary to clean its state.
                (when (si:scheduler-running-p)
                  (si::cleanup-scheduler-state))

		#+macosx
		(when (dolist (x (cdr sys::*command-line-arguments*))
			(when (and (> (length x) 4) (string= "-psn" x :end2 4))
			  (return t)))
		  (setq no-listener-sleep-forever t)
		  ;; Our CWD is /, so do something more friendly and go to
		  ;; the user's home directory:
		  (chdir))
		
                (setq *default-pathname-defaults* (current-directory))

                (let ((pll-file (internal-pll-file)))
                  (if* pll-file
                     then (setq *pll-file*
                                (merge-pathnames pll-file
                                                 *default-pathname-defaults*))
                     else (setq *pll-file* nil)))

                ;; HP compatibility does not extend between major releases.
                ;; The .dxl file is the same, but
                #+hpux
                (when (and (featurep :hpux-10)
                           (search "B.11." (software-version)))
                  (let ((name (get-shared-library-name)))
                    (warn "~
The ~s library was built on HP-UX 10.x, but is being run on HP-UX 11.x.  ~
Some libraries may not load properly; compatibility cannot be ~
guaranteed.  Use the HP-UX 11.0 ~s library instead." name name)))

                (reborn-ffi-init)

                (setq *internal-read-init-files* *read-init-files*)

                ;; Give internal pre-dumplisp code a chance to run any
                ;; cleanups.  This is used by, for example, SSL
                ;; initialization.  Using *restart-actions* prevent SSL use
                ;; from within init files.  THIS VARIABLE IS INTERNAL AND
                ;; SHOULD NOT BE USED.
                ;;
                ;; 2/1/07: This needs to be before command line processing
                ;; because acldns, pam and ssl use it to reinitialize their
                ;; subsystems during an image restart, and command line
                ;; processing can cause any one of these three modules to be
                ;; loaded and used.
                (mapc #'funcall (copy-list *system-restart-actions*))

                ;; Fixup the thread name.
                (setf (sys::thread-name (current-thread))
                      "Initial Lisp Listener")

                ;; This will always ignore args on initial build.
                ;; The command-line processing occurs early so it can
                ;; change the value of *internal-read-init-file*,
                ;; *restart-init-function*, or *restart-app-function*.
                (do-command-line-processing)

                ;; You are allowed to suppress (i.e., not display) the Franz
                ;; Inc. supplied copyright banner in your application, if and
                ;; only if you agree to display Franz Inc. copyright and
                ;; proprietary notices wherever you display your own copyright
                ;; or other proprietary notices.
                ;;
                ;; For further information see the :suppress-allegro-cl-banner
                ;; keyword argument to dumplisp.
                (when (null .dump-lisp-suppress-allegro-cl-banner.)
                  (copyright-banner))

                ;; Reset the external-format here since command-line
                ;; processing may have set *locale*, which, in turn,
                ;; affects the default external-format.
                (when (eq default-init-ef
                          (find-external-format
                           (stream-external-format *terminal-io*)))
                  (setf (stream-external-format *terminal-io*)
                        (find-external-format :default)))

                ;; If known, print the number of days until this Lisp expires.
                #+has-sys-expired
                (when (sys::expired)
                  (multiple-value-bind (second minute hour date month year)
                      (decode-universal-time
                       (+ (get-universal-time)
                          ;; seconds until we expire
                          (* *expire-days* #.(* 24 60 60))))
                    (declare (ignore second hour minute))
                    (let* ((month-name
                            (elt #("" "January" "February" "March" "April" "May"
                                   "June" "July" "August" "September" "October"
                                   "November" "December")
                                 month))
                           (pretty-date (format nil "~a ~d, ~d"
                                                month-name date year)))
                      (declare (special *expire-warning-date*))
                      (cond
                        ((featurep :allegro-cl-trial)
                         (when (and (numberp *expire-warning-date*)
                                    (> (get-universal-time)
                                       *expire-warning-date*))
                           (trial-expiration-warning pretty-date *expire-days*)))
                        (t (format
                            *stderr*
                            "Note: Allegro CL will expire in ~d days, on ~a.~%"
                            *expire-days* pretty-date)
                         (force-output *stderr*))))))

                ;; Will always skip reading on initial build.
                (read-init-files *internal-read-init-files*)

                (when (and *print-startup-message* reborn)
                  ;; [bug13978]: No longer need the handler-case:
                  (print-startup-info t))

                ;; Warns if *terminal-io* is not supporting 8-bit input.
                (ics-target-ecase
                 (:+ics (check-8-bit-input))
                 (:-ics nil))

                ;; Give pre-dumplisp code a chance to run any cleanups.  This
                ;; is the public restart actions.  SSL initialization, and
                ;; other internal programs, use *system-restart-actions* so
                ;; their initializations can be done before the init files are
                ;; read.
                (mapc #'funcall (copy-list *restart-actions*))

                ;; call this function to turn on the remote execution
                ;; interrupt lisp will process remote execution commands
                #+(target :alpha :alpha64 :hpprism :hp64 :linux86 :linux86-ot
                          :freebsd :msx86 :msdec :msx86-64 :sgi4d :sun4 :sparc-ot
                          :sparc64 :sparc64-ot :rs6000 :linux86-64 :macosx86-64)
                (excl::sig-handler-remote-execution nil nil)

                ;; This feature poses a security for applications that run
                ;; setuid root.
                #+ignore
                (let ((acl-startup-hook (sys:getenv "ACL_STARTUP_HOOK")))
                  (when acl-startup-hook
                    (ignore-errors (eval (read-from-string acl-startup-hook))))))))
      
      (check-for-pending-update-allegro)

      (when *restart-init-function*
        (funcall *restart-init-function*))

      #+mswindows
      ;; tell any interested outsider that we're ready for work
      (sig-handler-remote-execution nil nil)))
  
  ;; On Mac OS X as a GUI app, until we are a Cocoa app we can't do a
  ;; listener on stdin/stdout.
  (when no-listener-sleep-forever
    (loop (sleep most-positive-fixnum)))

  (tpl:start-interactive-top-level
   *initial-terminal-io*
   (or *restart-app-function* #'tpl:top-level-read-eval-print-loop)
   nil)
  ;; Now that we have a better handle on SIGHUP/SIGTERM handling, and
  ;; the ability to associate SIGPIPEs with particular streams, let's
  ;; try less of a panic. In particular, when any sort of socket or
  ;; terminal stream gets a sigpipe, the default action is to
  ;; change-class it to the null stream. This should protect pretty
  ;; well against recursive SIGPIPE lossage. The default handlers for
  ;; SIGHUP and SIGTERM immediately revert the signal action to the
  ;; default (which is to terminate the Unix process) so if a second
  ;; SIGTERM happens, lisp really will go away immediately. This was a
  ;; problem when running directly in an xterm (and perhaps other
  ;; environments) because when the xterm is told to go away, it sends
  ;; a SIGTERM to lisp, which starts shutting down in an orderly way
  ;; and eventually shuts down its streams. But as soon as its streams
  ;; are closed, the xterm sends _another_ SIGTERM (or maybe a SIGHUP)
  ;; which lisp used to try to report. But its streams were closed,
  ;; leading to recursive failure.
  (if* (si:scheduler-running-p)
     then (run-hooks *initial-lisp-listener-exit-hook*)
          (funcall-in-package :process-kill :multiprocessing
                              mp:*current-process*)
     else ;; Be careful to do no IO in case streams are already shut down
          ;; or otherwise broken.
          (exit-lisp -1 :quiet t)))

(defparameter *update-allegro-lock-file*
    "sys:UPDATEALLEGROPENDING.txt")

(defun check-for-pending-update-allegro ()
  (when (and (probe-file *update-allegro-lock-file*)
	     (null (sys:getenv "ACL_UPDATING_IMAGES")))
    (let ((message
	   (format nil
		   "~a must be run in ~a before Allegro will function properly"
		   #+mswindows "update.exe" #-mswindows "update.sh"
		   (translate-logical-pathname #p"sys:"))))
      #+mswindows (internal-message-box message)
      (error message))))

(defun trial-expiration-warning (pretty-date expire-days)
  (format
   t
   "~
NOTE: ********************************************************
NOTE: To continue uninterrupted use of Allegro CL
NOTE: you must obtain a new license before ~a
NOTE: (~d days from today). For more information visit:
NOTE:   http://www.franz.com/support/newlicense
NOTE: ********************************************************~%"
   pretty-date expire-days))

(defun reborn-ffi-init ()
  ;; On platforms where foreign functions are loaded (mapped)
  ;; dynamically, i.e. using .so files, and therefore not
  ;; written out with the dumplisp image, it is necessary
  ;; to reload all such libraries when the dumplisp is restarted.
  ;; This is done with the following call.  It is intended
  ;; that the path to .so files will be communicated using
  ;; logical pathnames.  If a .so file cannot be found, an
  ;; error break will happen which will give the user a
  ;; chance to supply a new pathname.  The reload has to
  ;; happen before any calls can be made to a foreign function
  ;; in a dynamically-loaded .so file. This is as good
  ;; a place as any to do the reloads, although the following
  ;; form could be moved slightly earlier or later if a
  ;; particular application makes additional
  ;; constaints on the restart procedures, for example, such
  ;; as needing foreign functions in the earlier
  ;; kill-processes-after-dumplisp processing, or wants
  ;; to defer reloading until after command-line processing.
  (if* (fboundp 'reload-fix-entry-points)
     then (reload-fix-entry-points (rebornp))
	  ;; [rfe6864]/[bug16921]
	  (relocate-callback-address-table)))

;; Some typical command-line argument processing:
;;  -q		don't load .clinit.cl in home directory
;;  -qq		don't load any .clinit.cl file.
;;  -e		(EVAL (READ)) of next arg.
;;  -batch		exit on any entry to the debugger.
;;  -backtrace-on-error
;;			do an auto zoom when an error occurs.

;; Remember that the emacs-lisp interface is typically started by a "-e"
;; argument, and this may call (use-background-streams).  Don't depend on
;; initial values of the standard stream variables at startup.

(defun do-command-line-processing ()
  (if* (null sys::.ignore-command-line-arguments.)
     then (do* ((xx (cdr sys::*command-line-arguments*) (cdr xx))
                (x (car xx)  (car xx))
                (y (cadr xx) (cadr xx)))
              ((or (null xx)
                   (string= "--" x)))
            (cond ((string= "--debug-startup" x)
                   (setq *debug-startup* t))
                  ((or (string= "-e" x)
                       (string= "--ee" x)
                       (string= "-ee" x))
                   (if* y
                      then (when (or (string= "--ee" x)
                                     (string= "-ee" x))
                             (setq y
                                   (nconvert-escaped-string y :escape #\%)))
                           (with-startup-phase
                               ("the reading or evaluation of -e ~s" y)
                             (eval (with-standard-io-syntax-and-readtable
                                     (read-from-string y))))
                           (setq xx (cdr xx))
                      else (warn "missing companion argument to -e")))
                  ((string= "-f" x)
                   (if* y
                      then (with-startup-phase
                               ("the reading or evaluation of -f ~s" y)
                             (funcall (with-standard-io-syntax-and-readtable
                                        (read-from-string y))))
                           (setq xx (cdr xx))
                      else (warn "missing companion argument to -f")))
                  ((string= "" x))
                  ((string= "-C" x)
                   (if* y
                      then (with-standard-io-syntax-and-readtable
                             (compile-file y))
                           (setq xx (cdr xx))
                      else (warn "missing companion argument to -C")))
                  ((string= "-L" x)
                   (if* y
                      then (with-startup-phase ("the load ~s" y)
                             (with-standard-io-syntax-and-readtable
                               (load y :verbose nil)))
                           (setq xx (cdr xx))
                      else (warn "missing companion argument to -L")))
                  ((or (string= "-#D" x)
                       (string= "-#C" x)
                       (string= "-#T" x)
                       (string= "-#!" x))
                   (setq *internal-read-init-files* nil)
                   (setq *print-startup-message* nil)
                   (setq sys::*application-command-line-arguments*
                         (cons (car sys::*command-line-arguments*)
                               (cddr xx)))
                   (setq .dump-lisp-suppress-allegro-cl-banner. t)
                   (setq *internal-read-init-files* nil)
                   (setq *force-quiet-exit* t)
                   (setq *load-print* nil)
                   (setq *load-verbose* nil)
                   (setq *compile-print* nil)
                   (setq *compile-verbose* nil)
                   (setq *global-gc-behavior* :auto)
                   (setq *batch-mode* t)
                   (setq *break-hook* #'exit-on-error-hook)
                   (when (null y) (error "Missing companion argument to -#"))
                   (setq *script* y)
                   (cond
                     ((string= "-#D" x) ;; debugging
                      (setq *backtrace-on-error* t)
                      (setq *restart-app-function*
                            (lambda ()
                              (tpl::read-eval-print-loop-wrapper
                               (lambda ()
                                 (load *script* :print nil :verbose nil
                                       :autoload t)
                                 (exit 0 :quiet t))))))
                     ((string= "-#C" x) ;; compile it
                      (setq *restart-app-function*
                            (lambda (&aux (status 0))
                              (tpl::read-eval-print-loop-wrapper
                               (lambda ()
                                 (handler-case
                                     (load (compile-file-if-needed
                                            *script* :script t :verbose nil
                                            :print nil)
                                           :print nil :verbose nil)
                                   (error (c)
                                     (format *error-output* "Error: ~a~%" c)
                                     (setq status 1)))
                                 (exit status :quiet t))))))
                     ((string= "-#T" x) ;; temp compile it
                      (let ((fasl
                             (compile-file-if-needed
                              *script* :script t :verbose nil :print nil
                              :output-file (system:make-temp-file-name
                                            "acl" "/tmp/"))))
                        (setq *restart-app-function*
                              (lambda (&aux (status 0))
                                (tpl::read-eval-print-loop-wrapper
                                 (lambda ()
                                   (unwind-protect
                                        (handler-case
                                            (load fasl :print nil :verbose nil)
                                          (error (c)
                                            (format *error-output* "Error: ~a~%" c)
                                            (setq status 0)))
                                     (ignore-errors (delete-file fasl)))
                                   (exit status :quiet t)))))))
                     (t ;; regular load
                      (setq *restart-app-function*
                            (lambda ()
                              (tpl::read-eval-print-loop-wrapper
                               (lambda (&aux (status 0))
                                 (handler-case
                                     (load *script* :print nil :verbose nil
                                           :autoload t)
                                   (error (c)
                                     (format *error-output* "Error: ~a~%" c)
                                     (setq status 1)))
                                 (exit status :quiet t)))))))
                   ;; rest of the args are for the script
                   (setq xx nil))
                  ((string= "-q" x)
                   (when *internal-read-init-files*
                     (setq *internal-read-init-files* :no-home)))
                  ((or (string= "--qq" x) (string= "-qq" x))
                   ;; This variable is reset nil each time lisp restarts.
                   (setq *internal-read-init-files* nil))
                  ((or (string= "--backtrace-on-error" x)
                       (string= "-backtrace-on-error" x))
                   (setq *backtrace-on-error* t)
                   (setq *break-hook* #'exit-on-error-hook))
                  ((or (string= "--bat" x) (string= "--bat-pause" x))
                   #+mswindows ;; noop on unix
                   (process-windows-bat-file (string= "--bat-pause" x) y)
                   (pop xx))
                  ((or (string= "--batch" x) (string= "-batch" x))
                   (setq *batch-mode* t)
                   (setq *break-hook* #'exit-on-error-hook))
                  ((or (string= "--kill" x) (string= "-kill" x))
                   (exit 0 :quiet t))
                  ((string= "-W" x)
                   (setq *break-on-warnings* t))
                  ((or (string= "-I" x)
                       (string= "-Im" x))
                   (setq *image-file* y)
                   (pop xx))
                  ((string= "-d" x)
                   #-ignore (dribble y)
                   ;; Not ready for prime time.  See details in rfe5838.
                   #+ignore
                   (progn
                     (setq *stderr*
                           (open y :direction :output
                                 :class 'line-buffered-file-stream
                                 :if-exists :supersede))
                     (stm-set-dribble-for-stream *terminal-io* *stderr*)
;;;; Before this can be used, we'll have to call dup2 in the runtime
;;;; system, since using it from the OSI module can't work.  In the typical
;;;; "make clean; make", osi.fasl will not exist when it is needed.
                     (funcall-in-package
                      :syscall-dup2 :excl.osi
                      (funcall-in-package :stream-to-fd :excl.osi *stderr*)
                      2))
                   (pop xx))
                  #-mswindows
                  ((or (string= "+s" x) (string= "-s" x))
                   (setq .startup-script. y)
                   (pop xx))
                  ((or
                    ;; rfe7485: restore -H, home location
                    (string= "-H" x)
                    ;; .lpr file for ide invocation
                    (string= "-project" x))
                   ;; skip over the associated value
                   (setq xx (cdr xx)))
                  ((or (string= "--locale" x) (string= "-locale" x))
                   (if* y
                      then (setq *locale*
                                 (locale-from-string y))
                      else (warn "Missing specifier after `-locale'"))
                   (setq xx (cdr xx)))
                  ((or (string="--compat-crlf" x)
                       (string="-compat-crlf" x))
                   (switch-to-crcrlf))
		  #+macosx
		  ((and (> (length x) 4) (string= "-psn" x :end2 4))
		   ;; ignore it
		   )
                  (t (warn "Unknown command-line option ~s" x)))))
  (setf sys::.ignore-command-line-arguments. nil))

#+mswindows
(defun process-windows-bat-file (pause bat-file)
  ;; Special batch file processing on Windows.
  ;; Skip everything until the first comment.
  ;; Read/eval everything until the keyword
  ;; :end-of-script is read, at which point we pause for N
  ;; seconds and exit.
  (setq *force-quiet-exit* t) ;; doesn't work 100%...
  (setq *load-print* nil)
  (setq *load-verbose* nil)
  (setq *compile-print* nil)
  (setq *compile-verbose* nil)
  (setq *global-gc-behavior* :auto)
  (flet ((pause ()
	   (when pause
	     (format t "~%~%Hit ENTER to exit:")
	     (force-output)
	     (read-char))))
    (handler-case
	(with-open-file (s bat-file :direction :input)
	  (let (line sexp)
	    (loop ;; skip everything up to and include the first lisp comment
	      (setq line (read-line s nil s))
	      (when (eq line s)
		;; shouldn't get here...
		(exit -1 :quiet t))
	      (when (and (> (length line) 0)
			 (char= #\; (schar line 0)))
		(return)))
	    (loop ;; process the script
	      (setq sexp (read s nil s))
	      (when (eq sexp s)
		;; shouldn't get here...
		(exit -1 :quiet t))
	      (when (eq :end-of-script sexp)
		(pause)
		(exit 0 :quiet t))
	      (eval sexp))))
      (error (c)
	(format t "~&~%Error processing ~a:~%~a" bat-file c)
	(terpri)
	(pause)
	(exit -1 :quiet t)))))

(defun lisp-to-bat-file (lisp-file bat-file
			 &key (executable "sys:mlisp.exe")
			      (image "sys:mlisp.dxl")
			      (pause t)
			      (if-exists :supersede))
  (with-open-file (in lisp-file :direction :input)
    (with-open-file (out bat-file :direction :output :if-exists if-exists)
      (format out "@echo off~%")
      (format out "\"~a\" -I \"~a\" ~a \"~a\" -- %*~%"
	      (translate-logical-pathname executable)
	      (translate-logical-pathname image)
	      (if pause "--bat-pause" "--bat")
	      (merge-pathnames bat-file))
      (format out "goto xxx_l2b_end~%")
      (format out ";;NEED THIS COMMENT TO DELINEATE THE LISP CODE~%")
      (sys:copy-file in out)
      (format out ":end-of-script~%")
      (format out ":xxx_l2b_end~%"))))

(defun convert-escaped-string (y &key (escape #\%))
  (nconvert-escaped-string (concatenate 'string y "") :escape escape))

(defun nconvert-escaped-string (y &key (escape #\%))
  ;; Frequently escaped chars:  ( %28  ) %29  : %3a  space %20  % %25
  (do* ((i 0) (ln (length y)) (newln ln))
      ((<= newln i) 
       (dotimes (j (- ln newln))
	 (setf (elt y (+ i j)) #\space)))
    (let* ((char (elt y i))
	   (d0 (when (< (1+ i) ln) (elt y (1+ i))))
	   (d1 (when (< (+ i 2) ln) (elt y (+ i 2))))
	   (hexl "0123456789abcdef")
	   (hexu "0123456789ABCDEF")
	   (n0 (when d0 (or (position d0 hexl) (position d0 hexu))))
	   (n1 (when d1 (or (position d1 hexl) (position d1 hexu)))))
      (when (and (eql char escape) n0 n1)
	(setf (elt y i) (code-char (+ n1 (* 16 n0))))
	(dotimes (j (- ln (+ i 3)))
	  (setf (elt y (+ i j 1)) (elt y (+ i j 3))))
	(decf newln 2))
      (incf i)))
  y)

;; Support for reading startup initialization files.

;; The name of the file read by the initial top-level upon startup.  It is
;; a list of files.  Since ".clinit.cl" is not an valid ISO 9660 filename
;; (used on CD-ROMs) allow "clinit.cl", too.  Only the first one, in a
;; given directory, is loaded.

(defvar *init-file-names* '(".clinit.cl" "clinit.cl"))

(defun read-init-files (read-init-files)
  (when (or (null read-init-files)
	    ;; Don't process command line args on initial build.
	    (eql *dumplisp-generation-tick* 0))
    (return-from read-init-files nil))
  (let ((site-init (ignore-errors (probe-file "sys:siteinit.cl"))))
    (when site-init
      (load-top-level-init-file site-init :verbose t)))
  (let ((current-pathname (current-directory))
	(u-hd-pathname (multiple-value-bind (success pn)
			   (errorset (user-homedir-pathname) t)
			 (when success pn))))
    ;; save the current pathname [which is the current directory
    ;; upon lisp startup] because it may change when the init file is read.
    ;; Load the user's home init file, then the one in the current directory.
    (if* u-hd-pathname
       then (if* (not (eq :no-home read-init-files))
	       then (load-top-level-init-file-from-directory u-hd-pathname)
		    ;; Make sure that we are not in the user's home directory,
		    ;; because we don't want to load the same file twice.
		    ;; Symbolic links on UNIX make this a bit tricky,
		    ;; so device/inode comparison is necessary.
		    ;; filesys-inode returns as two values the inode
		    ;; and dev numbers returned by Unix stat().
		    (and #+mswindows
			 (not
			  (namestring-compare
			   (namestring (truename u-hd-pathname))
			   (namestring (truename current-pathname))))
			 #-mswindows
			 (not
			  (equal (multiple-value-list
				     (filesys-inode
				      (namestring (truename u-hd-pathname))))
				 (multiple-value-list
				     (filesys-inode
				      (namestring
				       (truename current-pathname))))))
			 (load-top-level-init-file-from-directory
			  current-pathname))
	       else (load-top-level-init-file-from-directory current-pathname))
       else (warn "Skipping attempt to load any initialization files."))))

(defun load-top-level-init-file-from-directory (pathname)
  (dolist (init-file *init-file-names*)
    (setq init-file (merge-pathnames init-file pathname))
    (when (probe-file init-file)
      (load-top-level-init-file init-file)
      (return))))

(defun load-top-level-init-file (pathname &key verbose)
  (with-startup-phase ("loading init file ~s" pathname)
    (load pathname :verbose verbose :if-does-not-exist nil)))

(defvar .saved-break-level. nil)

(defvar *app-runtime-mode* nil)

(defun exit (&optional (code 0)
	     &key no-unwind (quiet *app-runtime-mode*))
  "This exits to the shell or OS, without querying the user, returning
STATUS to the program which invoked Common Lisp.  STATUS must be an
integer whose range is determined by the Operating System, and hence
the value may be truncated for `large' integers.

If NO-UNWIND is nil (the default), then `exit' executes all
outstanding unwind-protect cleanup forms for every process (see
mp:process-kill). If an error occurs during the execution of an
unwind-protect cleanup form or the process becomes inactive, then the
function `exit' could hang forever, waiting for the process to die.
The `exit' function prints the name of the process on which it is
waiting.  Processes which cause `exit' to hang indefinitely (because
the error cause the process to enter a break level) can be killed
manually from another window or editor buffer, or the `exit' can be
interrupted with ^C.  Note, if `exit' is executed again after being
interrupted, the unwind-protect which caused `exit' to hang will not
do so again, because the unwind-protect cleanup code is not
protected (unless it too contains an unwind-protect); for this reason,
`exit' could possibly be re-evaluated.

If NO-UNWIND is non-nil, then `exit' does not evaluate unwind-protect
cleanup forms."
  ;; As soon as we commit to exiting, revert SIGHUP and SIGTERM to the
  ;; default behavior to avoid loops.  If someone kicks us while we are
  ;; trying to leave, it's their fault if we don't close the door.
  (handler-case (when *dribble-active* (dribble))
    (error () nil))
  (unix-signal  1 0)
  (unix-signal 15 0)
  (flet ((exit-1 (&key no-unwind quiet)
	   (if* (boundp 'tpl::*break-level*)
	      then (setq .saved-break-level. tpl::*break-level*))
	   (if* (not (fixnump code)) then (setq code -1))

	   (let ((cleanup sys:*exit-cleanup-forms*))
	     ;; Clear the variable so if we get into trouble, exit will
	     ;; work the second time.
	     (setq sys:*exit-cleanup-forms* nil)
	     (dolist (x cleanup) (ignore-errors (eval x))))

	   (if* (si:scheduler-running-p)
	      then ;; This maybe unwinds and kills all but the current process.
		   (kill-processes-for-exit code no-unwind quiet))

	   (if* (not quiet) then  (force-output *terminal-io*))
	   (if* (not no-unwind)
	      then ;; We used to try to throw to non-mp-exit, because
	           ;; the initial lisp thread had a catch for that.
	           ;; There was only one thread with a catch for non-mp-exit
		   ;; and that was the initial thread.  We changed that
		   ;; catch to use the resume-thread tag and made
	           ;; the resume-thread catcher look for a second value
	           ;; telling it to exit (if non-nil) instead of maybe
	           ;; restarting the thread-function.
		   (let ((code code)
                         (quiet quiet))
                     (handler-case
                         (throw-lisp-kill code t quiet)
                       (condition () (exit-lisp code :quiet quiet)))))

	   (if* (not quiet) then (force-output *terminal-io*))
	   (force-output *stderr*) ;;rfe9199
	   (exit-lisp code :quiet quiet)))
    (restart-case (exit-1 :quiet quiet :no-unwind no-unwind)
      (nil ()
	  :report "Really exit (:no-unwind t)"
	(exit-1 :no-unwind t :quiet quiet)))))

(defvar *exit-lisp-announce-hook*
    (lambda ()
      (let ((s *initial-terminal-io*))
	(format s "; Exiting~%")
	#+ignore
	(let ((*terminal-io* s)
	      (*standard-output* s)
	      (tpl::*zoom-print-level* 10)
	      (tpl::*zoom-print-length* 10))
	  (tpl:do-command "zoom"
	    :from-read-eval-print-loop nil
	    :count t :all t))
	(force-output s))))

(defun exit-lisp (code &key quiet)
  ;; If :quiet is true, exit immediately doing no I/O.  For emergency
  ;; situations.
  (when (and *exit-lisp-announce-hook*
	     (not (or *force-quiet-exit* quiet)))
    (run-hooks *exit-lisp-announce-hook*))
  (excl::.primcall 'sys::lispexit
		   #+mswindows
		   (if* (and .saved-break-level.
			     (= code 0)
			     (> .saved-break-level. 0))
		      then .saved-break-level.
		      else code)
		   #-mswindows code)
  #+ignore
  (let ((s *initial-terminal-io*))
    (format s "; Failed Lisp Exit...~%")
    (force-output s)))

#-mswindows
(defun process-script (file)
  (with-open-file (s file :direction :input)
    (let (form val)
      (loop
	(setq form (read s nil s))
	(when (eq form s) (return))
	(setq val (multiple-value-list (eval form)))
	(if* (cdr val)
	   then (let ((i 1))
		  (dolist (v val)
		    (format t "~%value ~d: ~s~%" i v)
		    (incf i)))
	 elseif val
	   then (print (car val)))))))

(in-package :top-level)

(defun read-eval-print-loop-wrapper (func)
  (let* ((excl::*source-file-info* nil)
	 (excl::*recursive-read-list* t)
	 (*standard-input* *terminal-io*)
	 (*standard-output* *terminal-io*)
	 (*debug-io* *terminal-io*)
	 (*query-io* *terminal-io*)
	 (*trace-output* (if* excl::*trace-output-gravity* thenret else *terminal-io*))
	 (*error-output* *terminal-io*)
	 (*stepping* nil)
;;;;
	 (*unhide-all-frames* t)
	 (*print-readably* nil)
;;;;
	 (*focus-process* nil)		;the current process
	 (*focus-thread* nil)
	 (*tpl-current-thread* (excl::current-thread))
	 (*top-top-frame-pointer*
	  (excl::int-newest-frame (excl::current-thread)
				  :visible-only-p nil))
	 (*top-frame-pointer*
	  (or (unless (excl::package-not-yet-loaded :debugger)
		(excl::funcall-in-package :find-interesting-frame :debugger
					  *top-top-frame-pointer*))
	      *top-top-frame-pointer*))
	 (*current-frame-pointer* *top-frame-pointer*)
	 (sys::*compile-to-file* nil)
	 (sys::*compile-to-core* nil)
	 (*break-level* 1)
	 (*break-level-symbol* (excl::gentag))
	 *read-suppress*
	 excl::*circularity-hash-table*	;In case break happened during output.
	 )
    (declare (special excl::*recursive-read-list*))
    (funcall func)))
