;; -*- mode: common-lisp; package: excl -*-
;;
;; Support for the emacs-lisp interface.
;;
;; copyright (c) 1985, 1986 Franz Inc, Alameda, Ca.
;; copyright (c) 1986-2005 Franz Inc, Berkeley, CA  - All rights reserved.
;; copyright (c) 2002-2013 Franz Inc, Oakland, CA - All rights reserved.
;;
;; The software, data and information contained herein are proprietary
;; to, and comprise valuable trade secrets of, Franz, Inc.  They are
;; given in confidence by Franz, Inc. pursuant to a written license
;; agreement, and may be stored and used only in accordance with the terms
;; of such license.
;;
;; Restricted Rights Legend
;; ------------------------
;; Use, duplication, and disclosure of the software, data and information
;; contained herein by any agency, department or entity of the U.S.
;; Government are subject to restrictions of Restricted Rights for
;; Commercial Software developed at private expense as specified in
;; DOD FAR Supplement 52.227-7013 (c) (1) (ii), as applicable.
;;
;; $Id: eli.cl,v 1.32 2007/04/17 21:27:38 layer Exp $

(eval-when (compile eval load) (require :lep))

(eval-when (compile eval)
  (require :iodefs))

(in-package :excl)

(provide :eli)

;; [rfe6329]
#+process-autoloads
(autoload-from "code/misc.cl"
	       start-emacs-lisp-interface start-emacs-lisp-interface)

(defvar *ipc-version* 1
  ;; This indicates the version of the IPC protocol
  ;; For compatibility with the ipc.cl version of the Lisp side of the
  ;; Emacs interface.
  )

;; defvar for *emacs-daemon-password* moved to lep.cl, which uses it and
;; which this module requires. This eliminates a build-time warning.

(defparameter *eli-daemon-name*
    ;; Composer uses this, too.
    "Connect to Emacs daemon")

(defun new-start-emacs-lisp-interface (&key (background-streams t)
					    (listener-number 1)
					    port
					    announce-to-file)
  (load-emacs-mule-ef nil)		; bug14882
  (let ((proc (mp:process-run-function *eli-daemon-name*
		#'new-emacs-interface-daemon
		port
		announce-to-file)))
    (setf (mp::process-interruptible-p proc) nil)
    (setf (mp::process-keeps-lisp-alive-p proc) nil))
  (when background-streams (use-background-streams))
  (when listener-number
    (setf (getf (mp:process-property-list sys::*current-process*)
		:emacs-listener-number)
      listener-number))
  t)

(defparameter *eli-daemon-socket-hook*
    (lambda (port)
      (socket:make-socket :connect :passive :local-port port)))
;; How the above could be used to limit connections to the local machine:
#+ignore
(setq *eli-daemon-socket-hook*
  (lambda (port)
      (socket:make-socket :connect :passive
			  :local-port port
			  :local-host "127.0.0.1")))

(defun new-emacs-interface-daemon (&optional port announce-to-file)
  (let ((socket (funcall *eli-daemon-socket-hook* port)))
    (announce-server-startup-to-emacs (socket:local-port socket)
				      announce-to-file)
    (loop
      (let ((*read-eval* nil) ;; for security
	    (stream nil))
	(handler-case
	    (setq stream (socket:accept-connection socket))
	  (error (cond)
	    (format *initial-terminal-io*
		    "~
Accepting a connection from the emacs lisp interface resulted in error ~s"
		    cond)
	    (return-from new-emacs-interface-daemon)))
	#+debugging-only
	(format *initial-terminal-io*
		"~&Received connection from emacs at ~s~%"
		(socket:ipaddr-to-hostname (socket:remote-host stream)))
	(handler-case ;; the read's could cause an error
	    (start-emacs-process-for-network-stream (read stream)
						    stream)
	  (error (c)
	    (format *initial-terminal-io*
		    "~&Error (~a) starting emacs process.~%" c)
	    (close stream))))
      ;; Slow down a brute force password attack:
      (sleep 0.5))))

(defmethod start-emacs-process-for-network-stream ((protocol (eql :listener))
						   stream)
  (let* ((*read-eval* nil) ;; for security
	 (process-name (read stream))
	 (password (read stream)))
    (if* (not (and (numberp password)
		   (= password *emacs-daemon-password*)))
       then (format *initial-terminal-io*
		    "~&Access denied for ~s (password incorrect).~%"
		    (socket:ipaddr-to-hostname (socket:remote-host stream)))
	    (close stream)
       else (let ((proc
		   (mp:process-run-function process-name
					    #'eli-start-listener
					    stream)))
	      (setf (mp::process-interruptible-p proc) nil)
	      (setf (mp::process-keeps-lisp-alive-p proc) nil)))))

;; Stolen from ipc.cl:lisp-listener-with-stream-as-terminal-io:
(defun eli-start-listener (stream)
  (let ((our-fn
	 (with-stream-class (stream)
	   (sm input-handle stream))))
    (unwind-protect
	(handler-bind ((file-error
			#'(lambda (condition)
			    ;; see if the error is in
			    ;; this toplevel's stream and if so shut down
			    (if* (and (slot-exists-p
				       condition 'pathname)
				      (eql (slot-value condition
						       'pathname)
					   our-fn))
			       then (return-from eli-start-listener))))
		       (stream-closed-error
			#'(lambda (condition)
			    ;; in case someone shuts down our stream
			    ;; and thus we can't report about the error
			    (if* (eq (stream-error-stream condition)
				     stream)
			       then (return-from eli-start-listener)))))
	  ;; rfe5396: The following attempts to make Windows behave more
	  ;; like *nix, with respect to the behavior of where trace output
	  ;; goes for "background" streams.  On *nix it goes to the
	  ;; *common-lisp* buffer because that's the initial lisp
	  ;; listener.  On Windows, it used to go to the console window,
	  ;; but that's usually hidden or obscured.  Now, on Windows, it
	  ;; goes to the *common-lisp* buffer.
	  #+mswindows
	  (let ((x (assoc '*trace-output* *required-top-level-bindings* ;[rfe11065]
			  :test #'eq)))
	    (if* x
	       then ;; We are called multiple times, but the first time is
		    ;; for the *common-lisp* buffer--the run bar process
		    ;; goes through this code, too.
		    nil
	       else (push
		     (cons '*trace-output*
			   (system:global-symbol-value '*trace-output*))
		     *required-top-level-bindings*) ;[rfe11065]
		    (setq *trace-output* stream)))
	  
	  (tpl:start-interactive-top-level
	   stream 'tpl:top-level-read-eval-print-loop nil))
      ;; This next crock is to prevent the force-output done by close from
      ;; signalling an error if there are characters buffered to the output
      ;; stream, which there will be if the remote client closed the
      ;; connection.
      ;; This should be changed to a clear-output once that works on a
      ;; buffered terminal stream.
      (ignore-errors
       ;; ignore errors in the following two forms just in case the socket
       ;; gets messed up somehow--it has happened in customer reports, see
       ;; spr18345.
       (excl::clear-output-1 stream)
       (close stream)))))

(defun announce-server-startup-to-emacs (port announce-to-file)
  ;; We could use *standard-output* itself here, but the startup is
  ;; asynchronous with the initial lisp listener startup, and perturbing
  ;; the stream output column causes unpredictable indenting during
  ;; startup.  So instead we use a distinct stream that shares
  ;; *standard-output*'s fd.
  (let* ((*print-base* 10)
	 (announcement
	  (format nil "~d ~d ~a ~s ~d"
		  ;; ^Aport password case-mode socket-file ipc-version^A
		  port
		  (setq *emacs-daemon-password*
		    (random 1000000 (make-random-state t)))
		  (case excl::*current-case-mode*
		    ((:case-insensitive-upper
		      :case-sensitive-upper) ":upper")
		    ((:case-insensitive-lower
		      :case-sensitive-lower) ":lower"))
		  (list excl::cl-major-version-number
			excl::cl-minor-version-number
			excl::cl-version-type
			excl::cl-generation-number)
		  *ipc-version*)))
    (if* announce-to-file
       then (with-open-file (s announce-to-file :direction :output
			     :if-exists :supersede)
	      (write-string announcement s))
       else #+mswindows
	    (let ((file (merge-pathnames
			 (format nil "elistartup~d" (excl::getpid))
			 (sys:temporary-directory)))
		  (tmp (sys:make-temp-file-name)))
	      (with-open-file (s tmp :direction :output
			       :if-exists :supersede)
		(write-string announcement s))
	      ;; Rename the file to the final name to prevent race
	      ;; conditions (we know it's on the same filesystem, since
	      ;; make-temp-file-name uses temporary-directory 
	      (rename-file-raw tmp file))
	    #-mswindows
	    (let ((s (make-network-terminal-stream
		      (stream-output-fn *standard-output*))))
	      (write-string announcement s)
	      (finish-output s)))))

#-mswindows
(defun make-network-terminal-stream (fd &key pretty-id class element-type)
  (declare (ignore element-type))
  (let ((s (make-instance (or class
			      ;; 'excl::bidirectional-terminal-stream
			      ;; Go with simple-stream
			      'excl::terminal-simple-stream
			      )
	     ;; Go with simple-stream
	     ;; :element-type (or element-type 'character)
	     :input-handle  fd
	     :output-handle fd)))
    (when pretty-id
      (setf (getf (excl::stream-property-list s) :pretty-socket-id)
	pretty-id))
    s))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;; Compatibility with pre-ACL 7.0 versions of ELI
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun start-emacs-lisp-interface (&optional (use-background-streams t)
					     (emacs-listener-number 1)
					     port
					     announce-to-file)
  (load-emacs-mule-ef nil)
  (let ((proc (mp:process-run-function *eli-daemon-name*
				       #'emacs-interface-daemon
				       port
				       announce-to-file)))
    (setf (mp::process-interruptible-p proc) nil)
    (setf (mp::process-keeps-lisp-alive-p proc) nil))
  (when use-background-streams (use-background-streams))
  (when emacs-listener-number
    (setf (getf (mp:process-property-list sys::*current-process*)
		:emacs-listener-number)
      emacs-listener-number))
  t)

;; bug14283
(defun load-emacs-mule-ef (display-message)
  ;; This function is similar to calling (find-external-format :emacs-mule)
  ;; except that we don't try to load the external-format if we are in trial
  ;; (c.f. rfe6029).
  ;; If we run out of space during the load, then we try to clean up so that
  ;; Lisp can continue.
  ;;
  (or
   ;; Check to see if emacs-mule is already loaded without using
   ;; find-external-format.
   (and (fboundp 'all-external-formats)
	(find ':emacs-mule (all-external-formats)
	      :key #'(lambda (ef) (cons (ef-name ef) (ef-nicknames ef)))))
   ;; Call find-external-format inside a handler in case we need to clean up.
   (and (fboundp 'find-external-format)
	(handler-case
	    (if* (featurep ':allegro-cl-trial)
	       then (error 'storage-condition
			   :format-control "~
Not enough heap free to comfortably load emacs-mule external-format.")
	       else (find-external-format :emacs-mule
					  :errorp nil
					  :verbose nil))
	  ;; spr26554
	  (storage-condition (c)
	    (declare (ignore c))
	    ;; couldn't load in the external-format.  We
	    ;; are probably in a space-challenged trial release.
	    ;; Try recovering by gcing right away.  Most likely,
	    ;; we will not be doing this load attempt every time
	    ;; since the `provide' statement in the autoloaded
	    ;; file will already have been executed.
	    (declare (special excl::.emacs-mule-dim2-charsets.
			      excl::.unicode-to-emacs-mule-trie.
			      excl::.emacs-mule-dim1-charsets.))
	    (setq excl::.emacs-mule-dim2-charsets. nil
		  excl::.unicode-to-emacs-mule-trie. nil
		  excl::.emacs-mule-dim1-charsets. nil)
	    (gc)
	    (when display-message
	      (terpri *error-output*)
	      (note *error-output* "~
The emacs-mule external-format, used to transmit international (non-ascii) ~
characters for the emacs-lisp interface, is not being loaded due to lack of ~
heap space in this Lisp session."))
	    ;; indicate :emacs-mule not found.
	    nil)))))
  

(defparameter *emacs-interface-port-min*
    #+mswindows 9666
    #-mswindows 1025)

(defparameter *emacs-interface-port-max*
    #+mswindows 9666
    #-mswindows 2025)

(defun emacs-interface-daemon (&optional port announce-to-file)
  (declare (ignorable announce-to-file))
  (let* ((xport (or port *emacs-interface-port-min*))
	 (socket
	  (do ((s nil)
	       (reason nil))
	      ((or (handler-case
		       (setq s (funcall *eli-daemon-socket-hook* xport))
		     (error (c)
		       (setq reason (format nil "~a" c))
		       nil))
		   (= xport (or port *emacs-interface-port-max*)))
	       (or s (.error "Couldn't get a port: ~a" reason)))
	    (incf xport))))
    #-mswindows (announce-server-startup-to-emacs xport announce-to-file)
    (loop
      (let ((*read-eval* nil) ;; for security
	    (stream nil))
	(handler-case
	    (setq stream (socket:accept-connection socket))
	  (error (cond)
	    (format *initial-terminal-io*
		    "~
Accepting a connection from the emacs lisp interface resulted in error ~s"
		    cond)
	    (return-from emacs-interface-daemon)))
	(handler-case ;; the read's could cause an error
	    (start-emacs-process-for-network-stream (read stream) stream)
	  (error (c)
	    (format *initial-terminal-io*
		    "~&Error (~a) starting emacs process.~%" c)
	    (close stream))))
      ;; Slow down a brute force password attack:
      (sleep 0.5))))
